<?php

declare(strict_types=1);

namespace Erlage\Photogram;

use Exception;
use Throwable;
use Erlage\Photogram\Exceptions\RequestException;

final class SystemLogger
{
    public static function requestException(RequestException $exception): void
    {
        self::save('request exceptions', 'txt', $exception);
    }

    public static function serverException(Throwable $exception): void
    {
        self::detailedExceptionSave($exception, 'server');
    }

    public static function internalException(Throwable $exception): void
    {
        self::detailedExceptionSave($exception, 'internal');
    }

    public static function detailedExceptionSave(Throwable $exception, string $name): void
    {
        $data = array('Exception Dump' => (string) $exception);

        // try gathering more data
        try
        {
            $data['Exception'] = $exception -> getMessage();
            $data['Exception Stack Trace'] = $exception -> getTrace();
            $data['Request'] = Request::instance() -> getRequestData();
        }
        finally
        {
            self::save("{$name} exceptions", 'json', \json_encode($data, JSON_PRETTY_PRINT));
        }
    }

    public static function request(array $data): void
    {
        self::save('request logs', 'json', \json_encode($data, JSON_PRETTY_PRINT));
    }

    public static function response(array $data): void
    {
        self::save('response logs', 'json', \json_encode($data, JSON_PRETTY_PRINT));
    }

    public static function databaseAdapter(array $data): void
    {
        self::save('database', 'json', \json_encode($data, JSON_PRETTY_PRINT));
    }

    /**
     * 
     * @param string $type 
     * @param string $extension 
     * @param string|Exception $data 
     * @return void 
     * @throws Exception 
     */
    private static function save(string $type, string $extension, $data): void
    {
        $localHour = \date('g a');
        $localDay = \date('jS');
        $localMonth = \date('M');
        $localYear = \date('Y');

        $localFolders = array();

        $localLogPath = DIR_PATH_LOG;

        $localFolders[] = $localLogPath;
        $localFolders[] = $localLogPath . "{$type}/";
        $localFolders[] = $localLogPath . "{$type}/{$localMonth} {$localYear}";
        $localFolders[] = $localLogPath . "{$type}/{$localMonth} {$localYear}/{$localMonth} {$localDay} {$localYear}";
        $localFolders[] = $localLogPath . "{$type}/{$localMonth} {$localYear}/{$localMonth} {$localDay} {$localYear}/{$localHour}";

        foreach ($localFolders as $folderKey => $path)
        {
            if ( ! \file_exists($path))
            {
                /**
                 * Race conditions can occur here when there are too many requests
                 * So we're ignoring warning explicitly here
                 */
                try
                {
                    \mkdir($path);
                }
                catch (Exception $e)
                {
                    // nuffing
                }
            }
        }

        $localUniqueName = \microtime(true) . \random_int(111, 999);

        $localFullLogPath = $localFolders[4] . "/{$localYear} {$localMonth} {$localDay} {$localHour} {$localUniqueName}.{$extension}";

        \file_put_contents($localFullLogPath, $data);
    }
}
